import { NextResponse } from 'next/server';
import type { NextRequest } from 'next/server';
import { getSession } from '@/lib/auth';

export async function middleware(request: NextRequest) {
  const response = NextResponse.next();

  // 1. Security Headers
  response.headers.set('X-XSS-Protection', '1; mode=block');
  response.headers.set('X-Frame-Options', 'DENY'); // Prevent clickjacking
  response.headers.set('X-Content-Type-Options', 'nosniff');
  response.headers.set('Referrer-Policy', 'strict-origin-when-cross-origin');
  response.headers.set(
    'Content-Security-Policy',
    "default-src 'self'; script-src 'self' 'unsafe-inline' 'unsafe-eval' https://www.paypal.com https://www.sandbox.paypal.com; style-src 'self' 'unsafe-inline'; img-src 'self' blob: data: https://*.paypal.com; frame-src 'self' https://*.paypal.com;"
  );

  // 2. Protected Routes Logic (Optional - we already check in Pages/APIs, but double check here is good)
  // If user tries to access /admin without being admin, redirect to /
  if (request.nextUrl.pathname.startsWith('/admin')) {
    const session = await getSession(); // Note: getSession uses cookies(), might need edge compat
    // Edge runtime cookie parsing is tricky, for now rely on page/api checks or use simple cookie check
    const sessionCookie = request.cookies.get('session');
    if (!sessionCookie) {
        return NextResponse.redirect(new URL('/login', request.url));
    }
  }

  return response;
}

export const config = {
  matcher: [
    /*
     * Match all request paths except for the ones starting with:
     * - api (API routes)
     * - _next/static (static files)
     * - _next/image (image optimization files)
     * - favicon.ico (favicon file)
     */
    '/((?!api|_next/static|_next/image|favicon.ico).*)',
  ],
};
