import { connectToDatabase } from '@/lib/db';
import { User, License, Submission, Message, Tutorial, Update } from '@/lib/models';

// Helper to map filename to Mongoose Model
const MODEL_MAP: any = {
  'users.json': User,
  'licenses.json': License,
  'submissions.json': Submission,
  'messages.json': Message,
  'tutorials.json': Tutorial,
  'updates.json': Update
};

// READ: Fetch all documents
export async function readJSON<T>(filename: string): Promise<T> {
  await connectToDatabase();
  const Model = MODEL_MAP[filename];
  if (!Model) return [] as unknown as T;

  const docs = await Model.find({}).sort({ createdAt: -1 }).lean();
  
  // Map _id to id for frontend compatibility
  return docs.map((doc: any) => ({
    ...doc,
    id: doc._id.toString(),
    _id: undefined
  })) as unknown as T;
}

// WRITE: Overwrite is NOT efficient in MongoDB. 
// We should use specific CRUD operations instead of this generic "writeJSON".
// However, to keep the refactor simple for now, we will implement a "sync" strategy 
// or better yet, we will DEPRECATE this function and update the API routes to use Models directly.
// But for compatibility during migration, let's make it "save" the array? 
// NO, that's dangerous. 
// 
// STRATEGY: We will update the API routes to use Models directly.
// This function will throw an error to ensure we don't accidentally use it.
export async function writeJSON<T>(filename: string, data: T): Promise<void> {
  throw new Error("writeJSON is deprecated. Use Mongoose Models directly.");
}
