import mongoose, { Schema, model, models } from 'mongoose';

const UserSchema = new Schema({
  email: { type: String, required: true, unique: true },
  password: { type: String, required: true },
  role: { type: String, enum: ['user', 'admin', 'staff'], default: 'user' },
  isBanned: { type: Boolean, default: false },
  isMuted: { type: Boolean, default: false },
  isDisabled: { type: Boolean, default: false },
  lastLogin: { type: Date },
  createdAt: { type: Date, default: Date.now },
});

const LicenseSchema = new Schema({
  key: { type: String, required: true, unique: true },
  productSlug: { type: String, required: true },
  status: { type: String, enum: ['available', 'used'], default: 'available' },
  usedBy: { type: String }, // User ID
  orderId: { type: String },
  usedAt: { type: Date },
  createdAt: { type: Date, default: Date.now },
});

const SubmissionSchema = new Schema({
  userId: { type: String, required: true },
  userEmail: { type: String, required: true },
  productName: { type: String, required: true },
  productSlug: { type: String, required: true },
  amount: { type: Number, required: true },
  paymentMethod: { type: String, default: 'manual' },
  transactionId: { type: String, required: true },
  note: { type: String },
  adminNote: { type: String },
  receiptFilename: { type: String },
  status: { type: String, enum: ['pending', 'approved', 'rejected'], default: 'pending' },
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
});

const MessageSchema = new Schema({
  userId: { type: String, required: true },
  title: { type: String, required: true },
  content: { type: String, required: true },
  read: { type: Boolean, default: false },
  licenseKey: { type: String },
  productSlug: { type: String },
  createdAt: { type: Date, default: Date.now },
});

const TutorialSchema = new Schema({
  title: { type: String, required: true },
  description: { type: String, required: true },
  url: { type: String, required: true },
  icon: { type: String, default: 'Youtube' },
  color: { type: String, default: 'text-blue-400' },
});

const UpdateSchema = new Schema({
  version: { type: String, required: true },
  date: { type: String, required: true },
  changes: { type: [String], required: true },
});

const ChatSchema = new Schema({
  userId: { type: String, required: true },
  userEmail: { type: String, required: true },
  role: { type: String, required: true },
  isPremium: { type: Boolean, default: false },
  message: { type: String, required: true, maxlength: 200 },
  createdAt: { type: Date, default: Date.now },
});

const ProductSchema = new Schema({
  slug: { type: String, required: true, unique: true },
  name: { type: String, required: true },
  price: { type: Number, required: true },
  description: { type: String, required: true },
  image: { type: String },
  status: { type: String, enum: ['available', 'coming_soon'], default: 'available' },
  paypalLink: { type: String },
  createdAt: { type: Date, default: Date.now },
});

const ServiceSchema = new Schema({
  slug: { type: String, required: true, unique: true },
  title: { type: String, required: true },
  description: { type: String, required: true },
  priceRange: { type: String, required: true }, // e.g. "$5 - $50"
  image: { type: String },
  features: { type: [String], default: [] }, // List of what's included
  createdAt: { type: Date, default: Date.now },
});

const ServiceRequestSchema = new Schema({
  userId: { type: String, required: true },
  userEmail: { type: String, required: true },
  serviceSlug: { type: String, required: true },
  serviceTitle: { type: String, required: true },
  status: { type: String, enum: ['pending', 'negotiating', 'awaiting_payment', 'in_progress', 'completed', 'cancelled'], default: 'pending' },
  initialDetails: { type: String, required: true },
  agreedPrice: { type: Number },
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
});

const ServiceChatSchema = new Schema({
  requestId: { type: String, required: true }, // Links to ServiceRequest
  senderId: { type: String, required: true },
  senderRole: { type: String, required: true },
  message: { type: String, required: true },
  read: { type: Boolean, default: false },
  createdAt: { type: Date, default: Date.now },
});

export const User = models.User || model('User', UserSchema);
export const License = models.License || model('License', LicenseSchema);
export const Submission = models.Submission || model('Submission', SubmissionSchema);
export const Message = models.Message || model('Message', MessageSchema);
export const Tutorial = models.Tutorial || model('Tutorial', TutorialSchema);
export const Update = models.Update || model('Update', UpdateSchema);
export const Chat = models.Chat || model('Chat', ChatSchema);
export const Product = models.Product || model('Product', ProductSchema);
export const Service = models.Service || model('Service', ServiceSchema);
export const ServiceRequest = models.ServiceRequest || model('ServiceRequest', ServiceRequestSchema);
export const ServiceChat = models.ServiceChat || model('ServiceChat', ServiceChatSchema);
