'use client';

import { useEffect, useState } from 'react';
import Link from 'next/link';
import Image from 'next/image';
import { motion } from 'framer-motion';
import { ArrowRight, Check, Sparkles } from 'lucide-react';

export default function ServicesPage() {
  const [services, setServices] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetch('/api/services')
      .then(res => res.json())
      .then(data => {
        setServices(data.services || []);
        setLoading(false);
      });
  }, []);

  if (loading) return <div className="text-center py-20 text-white">Loading services...</div>;

  return (
    <div className="max-w-7xl mx-auto px-4 py-12">
       <div className="text-center mb-16">
        <h1 className="text-5xl font-extrabold mb-4 bg-gradient-to-r from-blue-400 via-purple-500 to-pink-500 bg-clip-text text-transparent animate-pulse">
            Custom Services
        </h1>
        <p className="text-gray-400 text-lg max-w-2xl mx-auto">
            Need something special? Our expert developers can build custom scripts, levels, and features tailored to your needs.
        </p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
        {services.map((service, index) => (
          <motion.div
            key={service.id}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: index * 0.1 }}
            className="bg-white/5 border border-white/10 rounded-2xl overflow-hidden hover:bg-white/10 transition-all group flex flex-col h-full"
          >
            {/* Image Section (Like Store) */}
            {service.image && (
              <div className="relative h-48 w-full overflow-hidden">
                <Image
                  src={service.image}
                  alt={service.title}
                  fill
                  className="object-cover group-hover:scale-110 transition-transform duration-500"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/80 to-transparent opacity-60" />
              </div>
            )}

            <div className="p-8 flex-1 pt-6">
                {/* Header with Icon if no image, or just title */}
                {!service.image && (
                    <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-purple-600 rounded-xl flex items-center justify-center mb-6 group-hover:scale-110 transition-transform shadow-lg shadow-purple-500/20">
                        <Sparkles className="text-white" />
                    </div>
                )}
                
                <div className="flex justify-between items-start mb-4">
                    <h2 className="text-2xl font-bold text-white">{service.title}</h2>
                    <span className="bg-white/10 backdrop-blur-md text-xs px-3 py-1 rounded-full text-white font-mono border border-white/5 h-fit whitespace-nowrap">
                        {service.priceRange}
                    </span>
                </div>

                <p className="text-gray-400 mb-6 leading-relaxed text-sm">
                    {service.description}
                </p>

                <ul className="space-y-3 mb-8">
                    {service.features.map((feature: string, i: number) => (
                        <li key={i} className="flex items-start gap-3 text-sm text-gray-300">
                            <Check size={16} className="text-green-400 mt-0.5 shrink-0" />
                            {feature}
                        </li>
                    ))}
                </ul>
            </div>

            <div className="p-6 pt-0 mt-auto">
                <Link 
                    href={`/services/${service.slug}`}
                    className="block w-full py-3 bg-white text-black font-bold text-center rounded-xl hover:bg-gray-200 transition-all flex items-center justify-center gap-2 group-hover:gap-3"
                >
                    Request Service <ArrowRight size={18} />
                </Link>
            </div>
          </motion.div>
        ))}
      </div>

      {services.length === 0 && (
          <div className="text-center text-gray-500">No services active at the moment. Check back later!</div>
      )}
    </div>
  );
}
