import { getSession } from '@/lib/auth';
import { readJSON } from '@/lib/store';
import { redirect } from 'next/navigation';

export default async function ProfilePage() {
  const session = await getSession();
  if (!session) {
    redirect('/login');
  }

  const submissions = await readJSON<any[]>('submissions.json');
  const userOrders = submissions
    .filter((s) => s.userId === session.user.id)
    .sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());

  const licenses = await readJSON<any[]>('licenses.json');
  const userLicenses = licenses.filter(l => l.usedBy === session.user.id);

  const totalSpent = userOrders
    .filter(o => o.status === 'approved')
    .reduce((acc, curr) => acc + (curr.amount || 0), 0);

  const accountStatus = userLicenses.length > 0 ? 'Premium' : 'Freemium';

  return (
    <div className="max-w-6xl mx-auto px-4 py-12">
      <h1 className="text-3xl font-bold mb-8">My Profile</h1>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* Sidebar / User Info */}
        <div className="space-y-8">
          <div className="bg-white/5 border border-white/10 rounded-2xl p-6">
            <div className="w-20 h-20 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-2xl font-bold text-white mb-4 mx-auto">
              {session.user.email[0].toUpperCase()}
            </div>
            <h2 className="text-xl font-bold text-center mb-1">{session.user.email}</h2>
            <p className="text-gray-400 text-sm text-center capitalize mb-6">{session.user.role}</p>
            
            <div className="border-t border-white/10 pt-4 space-y-3">
               <div className="flex justify-between text-sm">
                 <span className="text-gray-400">Member Since</span>
                 <span>{new Date(session.user.createdAt).toLocaleDateString()}</span>
               </div>
               <div className="flex justify-between text-sm">
                 <span className="text-gray-400">Total Spent</span>
                 <span className="text-green-400 font-bold">${totalSpent.toFixed(2)}</span>
               </div>
               <div className="flex justify-between text-sm">
                 <span className="text-gray-400">Account Status</span>
                 <span className={`px-2 py-0.5 rounded text-xs font-bold ${
                   accountStatus === 'Premium' ? 'bg-gradient-to-r from-yellow-400 to-orange-500 text-black' : 'bg-gray-700 text-gray-300'
                 }`}>{accountStatus}</span>
               </div>
            </div>
          </div>
        </div>

        {/* Main Content */}
        <div className="lg:col-span-2 space-y-8">
          
          {/* Licenses Section */}
          <div className="bg-white/5 border border-white/10 rounded-2xl p-6">
            <h3 className="text-xl font-bold mb-6 flex items-center gap-2">
              <span className="w-2 h-6 bg-green-500 rounded-full"></span>
              My Licenses
            </h3>
            
            {userLicenses.length === 0 ? (
              <div className="text-center py-8 text-gray-500">No active licenses found.</div>
            ) : (
               <div className="space-y-4">
                 {userLicenses.map((lic, i) => (
                   <div key={i} className="bg-black/20 border border-white/5 rounded-xl p-4">
                     <div className="flex justify-between items-start mb-2">
                       <span className="font-bold text-white">{lic.productSlug}</span>
                       <span className="text-xs text-gray-500">Activated: {new Date(lic.usedAt).toLocaleDateString()}</span>
                     </div>
                     <div className="bg-black/40 p-3 rounded border border-white/5 font-mono text-green-400 text-sm select-all">
                       {lic.key}
                     </div>
                   </div>
                 ))}
               </div>
            )}
          </div>

          {/* Order History Section */}
          <div className="bg-white/5 border border-white/10 rounded-2xl p-6">
            <h3 className="text-xl font-bold mb-6 flex items-center gap-2">
              <span className="w-2 h-6 bg-blue-500 rounded-full"></span>
              Order History
            </h3>
            
            {userOrders.length === 0 ? (
              <div className="text-center py-8 text-gray-500">No orders found.</div>
            ) : (
              <div className="overflow-x-auto">
                <table className="w-full text-left text-sm">
                  <thead className="text-gray-400 border-b border-white/10">
                    <tr>
                      <th className="pb-3 pl-2">Product</th>
                      <th className="pb-3">Date</th>
                      <th className="pb-3">Amount</th>
                      <th className="pb-3">Status</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-white/5">
                    {userOrders.map((order) => (
                      <tr key={order.id} className="group hover:bg-white/5 transition-colors">
                        <td className="py-4 pl-2 font-medium">{order.productName}</td>
                        <td className="py-4 text-gray-400">{new Date(order.createdAt).toLocaleDateString()}</td>
                        <td className="py-4">${order.amount}</td>
                        <td className="py-4">
                           <span className={`px-2 py-1 rounded text-xs font-bold uppercase ${
                            order.status === 'approved' ? 'bg-green-500/20 text-green-400' :
                            order.status === 'rejected' ? 'bg-red-500/20 text-red-400' :
                            'bg-yellow-500/20 text-yellow-400'
                           }`}>
                             {order.status}
                           </span>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>

        </div>
      </div>
    </div>
  );
}
