'use client';

import { useEffect, useState, useRef } from 'react';
import { useToast } from '@/app/context/ToastContext';
import { Send, MessageSquare, Trash2 } from 'lucide-react';

export default function CommunityPage() {
  const [messages, setMessages] = useState<any[]>([]);
  const [input, setInput] = useState('');
  const [loading, setLoading] = useState(true);
  const [sending, setSending] = useState(false);
  const [isAdmin, setIsAdmin] = useState(false); // Simple client-side check (server verifies anyway)
  const chatBoxRef = useRef<HTMLDivElement>(null);
  const { addToast } = useToast();

  // Check user role
  useEffect(() => {
    fetch('/api/auth/me')
      .then(res => res.json())
      .then(data => {
        if (data.user && (data.user.role === 'admin' || data.user.role === 'staff')) {
            setIsAdmin(true);
        }
      })
      .catch(() => {});
  }, []);

  // Fetch messages
  const fetchMessages = async () => {
    try {
      const res = await fetch('/api/chat');
      const data = await res.json();
      if (data.messages) {
        setMessages(data.messages);
      }
      setLoading(false);
    } catch (error) {
      console.error("Chat error:", error);
    }
  };

  // Poll for new messages every 3 seconds
  useEffect(() => {
    fetchMessages();
    const interval = setInterval(fetchMessages, 3000);
    return () => clearInterval(interval);
  }, []);

  // Auto-scroll to bottom on new messages
  useEffect(() => {
    if (chatBoxRef.current) {
      chatBoxRef.current.scrollTop = chatBoxRef.current.scrollHeight;
    }
  }, [messages]);

  const handleSend = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!input.trim()) return;

    setSending(true);
    try {
      const res = await fetch('/api/chat', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ message: input })
      });

      const data = await res.json();

      if (!res.ok) {
        addToast(data.error || "Failed to send", 'error');
      } else {
        setInput('');
        fetchMessages(); // Instant refresh
      }
    } catch (error) {
      addToast("Network error", 'error');
    } finally {
      setSending(false);
    }
  };

  const handleDelete = async (id: string) => {
    if (!confirm("Delete this message?")) return;
    try {
        const res = await fetch('/api/chat/delete', {
            method: 'DELETE',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ id })
        });
        if (res.ok) {
            addToast("Message deleted", 'success');
            fetchMessages();
        } else {
            addToast("Failed to delete", 'error');
        }
    } catch (e) {
        addToast("Error deleting", 'error');
    }
  };

  return (
    <div className="max-w-6xl mx-auto px-4 py-12 h-[calc(100vh-4rem)] flex flex-col">
      <div className="flex justify-between items-end mb-6">
        <div>
          <h1 className="text-3xl font-bold bg-gradient-to-r from-purple-400 to-pink-600 bg-clip-text text-transparent">
            Community
          </h1>
          <p className="text-gray-400">Chat with other users and staff.</p>
        </div>
        
        <a 
          href="https://discord.gg/Q4fsNPPnSY" 
          target="_blank" 
          rel="noopener noreferrer"
          className="bg-[#5865F2] hover:bg-[#4752C4] text-white px-4 py-2 rounded-lg font-bold flex items-center gap-2 transition-colors"
        >
          <MessageSquare size={20} />
          Join Discord
        </a>
      </div>

      <div className="flex-1 bg-white/5 border border-white/10 rounded-2xl overflow-hidden flex flex-col">
        {/* Chat Area */}
        <div 
          ref={chatBoxRef}
          className="flex-1 overflow-y-auto p-6 space-y-4 scroll-smooth"
        >
          {loading ? (
            <div className="text-center text-gray-500 mt-10">Loading chat...</div>
          ) : messages.length === 0 ? (
            <div className="text-center text-gray-500 mt-10">No messages yet. Be the first!</div>
          ) : (
            messages.map((msg) => (
              <div key={msg.id} className="flex flex-col group relative">
                <div className="flex items-baseline gap-2">
                  <span className={`font-bold text-sm ${
                    msg.role === 'admin' ? 'text-transparent bg-clip-text bg-gradient-to-r from-red-500 via-purple-500 to-blue-500 animate-pulse' :
                    msg.isPremium ? 'text-transparent bg-clip-text bg-gradient-to-r from-yellow-400 to-orange-500' :
                    msg.role === 'staff' ? 'text-purple-400' : 
                    'text-blue-400'
                  }`}>
                    {msg.role === 'admin' ? '[OWNER] ' : msg.role === 'staff' ? '[STAFF] ' : msg.isPremium ? '[PREMIUM] ' : ''}
                    {msg.userEmail.split('@')[0]}
                  </span>
                  <span className="text-[10px] text-gray-600">
                    {new Date(msg.createdAt).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                  </span>
                  
                  {isAdmin && (
                    <div className="flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity ml-2">
                      <button 
                        onClick={() => handleDelete(msg.id)}
                        className="text-red-500 hover:text-red-400"
                        title="Delete Message"
                      >
                        <Trash2 size={12} />
                      </button>
                      <button 
                        onClick={() => {
                           if(confirm("Delete ALL messages from this user?")) {
                               fetch('/api/chat/delete', {
                                    method: 'DELETE',
                                    headers: { 'Content-Type': 'application/json' },
                                    body: JSON.stringify({ userId: msg.userId })
                                }).then(() => {
                                    addToast("Nuked user messages", 'success');
                                    fetchMessages();
                                });
                           }
                        }}
                        className="text-red-700 hover:text-red-500 font-bold text-[10px]"
                        title="Delete ALL from user"
                      >
                        NUKE
                      </button>
                    </div>
                  )}
                </div>
                <p className="text-gray-300 text-sm break-words bg-black/20 p-2 rounded-lg rounded-tl-none mt-1 inline-block max-w-[80%]">
                  {msg.message}
                </p>
              </div>
            ))
          )}
        </div>

        {/* Input Area */}
        <form onSubmit={handleSend} className="p-4 bg-black/20 border-t border-white/10 flex gap-2">
          <input
            type="text"
            value={input}
            onChange={(e) => setInput(e.target.value)}
            placeholder="Type a message..."
            maxLength={200}
            className="flex-1 bg-white/5 border border-white/10 rounded-xl px-4 py-2 focus:outline-none focus:border-white/30 transition-colors"
            disabled={sending}
          />
          <button 
            type="submit" 
            disabled={sending || !input.trim()}
            className="bg-blue-600 hover:bg-blue-500 text-white p-2 rounded-xl disabled:opacity-50 transition-colors"
          >
            <Send size={20} />
          </button>
        </form>
      </div>
    </div>
  );
}
