import { NextResponse } from 'next/server';
import { getSession } from '@/lib/auth';
import { connectToDatabase } from '@/lib/db';
import { ServiceRequest, ServiceChat } from '@/lib/models';

export async function POST(request: Request) {
  const session = await getSession();
  if (!session) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
  }

  const { requestId, transactionId } = await request.json();

  await connectToDatabase();
  const req = await ServiceRequest.findById(requestId);

  if (!req) return NextResponse.json({ error: "Request not found" }, { status: 404 });

  if (req.status !== 'awaiting_payment') {
      return NextResponse.json({ error: "Invalid status" }, { status: 400 });
  }

  // Verify ownership (optional double check)
  if (req.userId !== session.user.id) {
      return NextResponse.json({ error: "Forbidden" }, { status: 403 });
  }

  // Update status
  req.status = 'in_progress';
  req.updatedAt = new Date();
  // In a real app, you might save transactionId somewhere too
  await req.save();

  // System message
  await ServiceChat.create({
      requestId: requestId,
      senderId: 'system',
      senderRole: 'system',
      message: `Payment successful (ID: ${transactionId}). Status updated to IN PROGRESS.`
  });

  return NextResponse.json({ success: true });
}
