import { NextResponse } from 'next/server';
import { connectToDatabase } from '@/lib/db';
import { Product } from '@/lib/models';
import { getSession } from '@/lib/auth';

export async function GET() {
  await connectToDatabase();
  const products = await Product.find({}).sort({ createdAt: -1 }).lean();
  
  const mappedProducts = products.map((p: any) => ({
    ...p,
    id: p._id.toString(),
    _id: undefined
  }));

  return NextResponse.json({ products: mappedProducts });
}

export async function POST(request: Request) {
  const session = await getSession();
  if (!session || (session.user.role !== 'admin' && session.user.role !== 'staff')) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 403 });
  }

  const body = await request.json();
  const { slug, name, price, description, image, status, paypalLink } = body;

  if (!slug || !name || price === undefined || !description) {
      return NextResponse.json({ error: "Missing required fields" }, { status: 400 });
  }

  await connectToDatabase();

  // Check duplicate slug
  const existing = await Product.findOne({ slug });
  if (existing) {
      return NextResponse.json({ error: "Product slug already exists" }, { status: 400 });
  }

  const newProduct = await Product.create({
    slug,
    name,
    price,
    description,
    image,
    status,
    paypalLink
  });

  return NextResponse.json({ 
      success: true, 
      product: { 
          ...newProduct.toObject(), 
          id: newProduct._id.toString(), 
          _id: undefined 
      } 
  });
}
