import { NextResponse } from 'next/server';
import { getSession } from '@/lib/auth';
import { connectToDatabase } from '@/lib/db';
import { ServiceChat, ServiceRequest } from '@/lib/models';

export const dynamic = 'force-dynamic';

export async function GET() {
  const session = await getSession();
  if (!session) {
    return NextResponse.json({ count: 0 });
  }

  await connectToDatabase();

  let unreadCount = 0;

  if (session.user.role === 'admin' || session.user.role === 'staff') {
      // Admins see messages from USERS that are unread
      // We need to find all chats where senderRole is NOT admin/staff/system AND read is false
      // Actually, simpler: find chats where senderId != session.user.id AND read is false
      // But admins act as a group. So maybe messages where senderRole == 'user'?
      unreadCount = await ServiceChat.countDocuments({
          senderRole: 'user',
          read: false
      });
  } else {
      // Users see messages from ADMINS/STAFF/SYSTEM that are unread
      // Find chats linked to requests owned by this user?
      // Step 1: Find all request IDs owned by user
      const userRequests = await ServiceRequest.find({ userId: session.user.id }).select('_id');
      const requestIds = userRequests.map(r => r._id.toString());

      // Step 2: Count unread messages in those requests sent by NON-users (admin/staff/system)
      unreadCount = await ServiceChat.countDocuments({
          requestId: { $in: requestIds },
          senderRole: { $ne: 'user' },
          read: false
      });
  }

  return NextResponse.json({ count: unreadCount });
}
