import { NextResponse } from 'next/server';
import { getSession } from '@/lib/auth';
import { connectToDatabase } from '@/lib/db';
import { Chat, License } from '@/lib/models';
import { rateLimit } from '@/lib/ratelimit';

export async function GET() {
  await connectToDatabase();
  // Fetch last 50 messages
  const messages = await Chat.find({}).sort({ createdAt: -1 }).limit(50).lean();
  
  // Return in chronological order (oldest to newest) for UI
  return NextResponse.json({ 
    messages: messages.reverse().map((m: any) => ({
        ...m,
        id: m._id.toString(),
        _id: undefined
    })) 
  });
}

export async function POST(request: Request) {
  const session = await getSession();
  if (!session) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
  }

  // Rate Limit: 10 messages per minute
  const ip = request.headers.get('x-forwarded-for') || 'unknown';
  if (!rateLimit(ip, 'chat', 10, 60 * 1000)) {
    return NextResponse.json({ error: "You are chatting too fast." }, { status: 429 });
  }

  const { message } = await request.json();

  // BAD WORD FILTER
  const badWords = ["porn", "xxx", "nude", "sex", "hentai"];
  if (badWords.some(word => message.toLowerCase().includes(word))) {
     return NextResponse.json({ error: "Message contains restricted words." }, { status: 400 });
  }

  if (!message || typeof message !== 'string' || message.trim().length === 0) {
    return NextResponse.json({ error: "Message cannot be empty" }, { status: 400 });
  }

  if (message.length > 200) {
    return NextResponse.json({ error: "Message too long (max 200 chars)" }, { status: 400 });
  }

  await connectToDatabase();

  // Check if user is premium (has at least one license)
  const licenseCount = await License.countDocuments({ usedBy: session.user.id });
  const isPremium = licenseCount > 0;

  const newChat = await Chat.create({
    userId: session.user.id,
    userEmail: session.user.email,
    role: session.user.role,
    isPremium,
    message: message.trim()
  });

  return NextResponse.json({ success: true, message: newChat });
}
