import { NextResponse } from 'next/server';
import bcrypt from 'bcryptjs';
import { connectToDatabase } from '@/lib/db';
import { User } from '@/lib/models';
import { login } from '@/lib/auth';
import { rateLimit } from '@/lib/ratelimit';
import { z } from 'zod';
import { cookies } from 'next/headers';

const registerSchema = z.object({
  email: z.string().email(),
  password: z.string().min(6, "Password must be at least 6 characters")
});

export async function POST(request: Request) {
  try {
    const ip = request.headers.get('x-forwarded-for') || 'unknown';
    
    // 1. IP Rate Limiting (existing)
    if (!rateLimit(ip, 'register', 3, 60 * 60 * 1000)) {
      return NextResponse.json({ error: "Too many registration attempts. Please try again later." }, { status: 429 });
    }

    // 2. Device Cookie Check (Anti-Alt)
    // Just block them if they have the cookie "registered_recently"
    const recentReg = cookies().get('registered_recently');
    if (recentReg) {
            return NextResponse.json({ error: "You have already registered an account recently." }, { status: 429 });
    }

    const body = await request.json();
    
    // Validation
    const result = registerSchema.safeParse(body);
    if (!result.success) {
        return NextResponse.json({ error: result.error.issues[0].message }, { status: 400 });
    }
    const { email, password } = result.data;

    await connectToDatabase();

    // Check existing
    const existingUser = await User.findOne({ email });
    if (existingUser) {
      return NextResponse.json({ error: "User already exists" }, { status: 400 });
    }

    const hashedPassword = await bcrypt.hash(password, 10);
    
    // First user is admin logic
    const userCount = await User.countDocuments();
    const role = userCount === 0 ? 'admin' : 'user';

    const newUser = await User.create({
      email,
      password: hashedPassword,
      role,
    });

    // Set "registered_recently" cookie for 1 hour
    cookies().set('registered_recently', 'true', { 
        expires: new Date(Date.now() + 60 * 60 * 1000), 
        httpOnly: true,
        secure: process.env.NODE_ENV === 'production'
    });

    // No auto-login
    
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error("Register error:", error);
    return NextResponse.json({ error: "Internal server error" }, { status: 500 });
  }
}
