import { NextResponse } from 'next/server';
import { getSession } from '@/lib/auth';
import { connectToDatabase } from '@/lib/db';
import { Service } from '@/lib/models';

export async function POST(request: Request) {
  const session = await getSession();
  if (!session || (session.user.role !== 'admin' && session.user.role !== 'staff')) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 403 });
  }

  try {
    const { title, description, priceRange, slug, features, image } = await request.json();

    // Validate required fields
    if (!title || !slug || !priceRange) {
        return NextResponse.json({ error: "Missing required fields (title, slug, priceRange)" }, { status: 400 });
    }

    await connectToDatabase();

    // Check duplicate slug
    const existing = await Service.findOne({ slug });
    if (existing) {
        return NextResponse.json({ error: "Slug already exists" }, { status: 400 });
    }

    const newService = await Service.create({
        title,
        description,
        priceRange,
        slug,
        image,
        features: features || []
    });

    return NextResponse.json({ success: true, service: newService });
  } catch (error: any) {
      console.error("Create Service Error:", error);
      return NextResponse.json({ error: error.message || "Internal Server Error" }, { status: 500 });
  }
}

export async function DELETE(request: Request) {
    const session = await getSession();
    if (!session || (session.user.role !== 'admin' && session.user.role !== 'staff')) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 403 });
    }
  
    const { id } = await request.json();
    await connectToDatabase();
    await Service.findByIdAndDelete(id);
  
    return NextResponse.json({ success: true });
  }
