import { NextResponse } from 'next/server';
import { getSession } from '@/lib/auth';
import { connectToDatabase } from '@/lib/db';
import { License, Message, User, Submission } from '@/lib/models';

export async function POST(request: Request) {
  const session = await getSession();
  if (!session || (session.user.role !== 'admin' && session.user.role !== 'staff')) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 403 });
  }

  const { email, productSlug, productName, transactionId } = await request.json();

  if (!email || !productSlug) {
      return NextResponse.json({ error: "Email and Product Slug required" }, { status: 400 });
  }

  await connectToDatabase();

  // 1. Find User
  const user = await User.findOne({ email });
  if (!user) {
      return NextResponse.json({ error: "User not found with that email" }, { status: 404 });
  }

  // 2. Find Available License
  const license = await License.findOne({ productSlug, status: 'available' });
  if (!license) {
      return NextResponse.json({ error: "No available license keys for this product" }, { status: 404 });
  }

  // 3. Create a "Fake" Submission (to track the order history)
  // We check if one exists first to avoid duplicates if possible, but for manual assign, we just create one.
  const submission = await Submission.create({
      userId: user._id.toString(),
      userEmail: user.email,
      productName: productName || productSlug,
      productSlug: productSlug,
      amount: 0, // Manual assignment
      paymentMethod: 'manual_admin',
      transactionId: transactionId || 'MANUAL_ASSIGN_' + Date.now(),
      status: 'approved',
      adminNote: 'Manually assigned by admin',
      updatedAt: new Date()
  });

  // 4. Assign License
  license.status = 'used';
  license.usedBy = user._id.toString();
  license.usedAt = new Date();
  license.orderId = submission._id.toString();
  await license.save();

  // 5. Send Message to User (The "System" delivery)
  const messageContent = `Your order for ${productName || productSlug} has been manually confirmed by an admin.\n\nHere is your License Key:\n${license.key}\n\nPlease keep this key safe.`;

  await Message.create({
    userId: user._id.toString(),
    title: 'Order Confirmed: ' + (productName || productSlug),
    content: messageContent,
    licenseKey: license.key, 
    productSlug: productSlug
  });

  return NextResponse.json({ success: true, key: license.key });
}
