'use client';

import { useEffect, useState } from 'react';
import { useToast } from '@/app/context/ToastContext';

export default function UsersPage() {
  const [users, setUsers] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const { addToast } = useToast();

  const fetchUsers = () => {
    fetch('/api/admin/users/list')
      .then((res) => res.json())
      .then((data) => {
        setUsers(data.users || []);
        setLoading(false);
      })
      .catch(() => addToast("Failed to fetch users", 'error'));
  };

  useEffect(() => {
    fetchUsers();
  }, []);

  const handleAction = async (id: string, action: string, value: any) => {
    const res = await fetch(`/api/admin/users/${id}/update`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ action, value })
    });

    if (res.ok) {
      addToast(`User updated successfully`, 'success');
      // Optimistic update
      setUsers(prev => prev.map(u => {
        if (u.id === id) {
            if (action === 'ban') return { ...u, isBanned: value };
            if (action === 'mute') return { ...u, isMuted: value };
            if (action === 'disable') return { ...u, isDisabled: value };
            if (action === 'role') return { ...u, role: value };
        }
        return u;
      }));
    } else {
      const data = await res.json();
      addToast(data.error || "Update failed", 'error');
    }
  };

  if (loading) return <div>Loading users...</div>;

  return (
    <div className="max-w-6xl mx-auto px-4 py-12">
      <h1 className="text-3xl font-bold mb-8">User Management</h1>

      <div className="bg-white/5 border border-white/10 rounded-xl overflow-hidden">
        <table className="w-full text-left text-sm">
          <thead className="bg-black/20 text-gray-400">
            <tr>
              <th className="px-6 py-3">Email</th>
              <th className="px-6 py-3">Role</th>
              <th className="px-6 py-3">Joined</th>
              <th className="px-6 py-3">Last Login</th>
              <th className="px-6 py-3 text-right">Actions</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-white/5">
            {users.map((user) => (
              <tr key={user.id} className="hover:bg-white/5">
                <td className="px-6 py-4">
                  <div className="font-medium">{user.email}</div>
                  <div className="text-xs text-gray-500 space-x-2">
                    {user.isBanned && <span className="text-red-400 font-bold">BANNED</span>}
                    {user.isDisabled && <span className="text-gray-400 font-bold">DISABLED</span>}
                    {user.isMuted && <span className="text-yellow-400 font-bold">MUTED</span>}
                  </div>
                </td>
                <td className="px-6 py-4">
                  <select 
                    value={user.role}
                    onChange={(e) => handleAction(user.id, 'role', e.target.value)}
                    className="bg-black/20 border border-white/10 rounded px-2 py-1 text-xs"
                  >
                    <option value="user" className="bg-black">User</option>
                    <option value="staff" className="bg-black">Staff</option>
                    <option value="admin" className="bg-black">Admin</option>
                  </select>
                </td>
                <td className="px-6 py-4 text-gray-500">{new Date(user.createdAt).toLocaleDateString()}</td>
                <td className="px-6 py-4 text-gray-500">{user.lastLogin ? new Date(user.lastLogin).toLocaleDateString() : 'Never'}</td>
                <td className="px-6 py-4 text-right space-x-2">
                  <button 
                    onClick={() => handleAction(user.id, 'ban', !user.isBanned)}
                    className={`px-3 py-1 rounded text-xs font-bold transition-colors ${user.isBanned ? 'bg-red-500 text-white' : 'bg-white/5 hover:bg-red-500/20 text-red-400'}`}
                  >
                    {user.isBanned ? 'Unban' : 'Ban'}
                  </button>
                  <button 
                    onClick={() => handleAction(user.id, 'mute', !user.isMuted)}
                    className={`px-3 py-1 rounded text-xs font-bold transition-colors ${user.isMuted ? 'bg-yellow-500 text-black' : 'bg-white/5 hover:bg-yellow-500/20 text-yellow-400'}`}
                  >
                    {user.isMuted ? 'Unmute' : 'Mute'}
                  </button>
                  <button 
                    onClick={() => handleAction(user.id, 'disable', !user.isDisabled)}
                    className={`px-3 py-1 rounded text-xs font-bold transition-colors ${user.isDisabled ? 'bg-gray-500 text-white' : 'bg-white/5 hover:bg-gray-500/20 text-gray-400'}`}
                  >
                    {user.isDisabled ? 'Enable' : 'Disable'}
                  </button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}
