'use client';

import { useEffect, useState } from 'react';
import { useToast } from '@/app/context/ToastContext';

export default function ManageTutorialsPage() {
  const [tutorials, setTutorials] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [formData, setFormData] = useState({ title: '', description: '', url: '', icon: 'Youtube', color: 'text-red-400' });
  const { addToast } = useToast();

  const fetchTutorials = () => {
    fetch('/api/tutorials')
      .then((res) => res.json())
      .then((data) => {
        setTutorials(data.tutorials || []);
        setLoading(false);
      })
      .catch(() => addToast("Failed to fetch tutorials", 'error'));
  };

  useEffect(() => {
    fetchTutorials();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    await fetch('/api/tutorials', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(formData)
    });
    setFormData({ title: '', description: '', url: '', icon: 'Youtube', color: 'text-red-400' });
    fetchTutorials();
    addToast("Tutorial added successfully", 'success');
  };

  const handleDelete = async (id: string) => {
    if(!confirm("Delete this tutorial?")) return;
    await fetch(`/api/tutorials/${id}`, { method: 'DELETE' });
    fetchTutorials();
    addToast("Tutorial deleted", 'info');
  };

  if (loading) return <div>Loading...</div>;

  return (
    <div className="max-w-6xl mx-auto px-4 py-12">
      <h1 className="text-3xl font-bold mb-8">Manage Tutorials</h1>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <div className="bg-white/5 border border-white/10 rounded-xl p-6 h-fit">
           <h2 className="text-xl font-bold mb-4">Add New Tutorial</h2>
           <form onSubmit={handleSubmit} className="space-y-4">
             <input 
                type="text" placeholder="Title" className="w-full bg-black/20 border border-white/10 rounded p-2"
                value={formData.title} onChange={e => setFormData({...formData, title: e.target.value})} required
             />
             <textarea 
                placeholder="Description" className="w-full bg-black/20 border border-white/10 rounded p-2"
                value={formData.description} onChange={e => setFormData({...formData, description: e.target.value})} required
             />
             <input 
                type="url" placeholder="YouTube URL" className="w-full bg-black/20 border border-white/10 rounded p-2"
                value={formData.url} onChange={e => setFormData({...formData, url: e.target.value})} required
             />
             <select 
                className="w-full bg-black/20 border border-white/10 rounded p-2"
                value={formData.icon} onChange={e => setFormData({...formData, icon: e.target.value})}
             >
                <option value="Youtube" className="bg-black">Youtube Icon</option>
                <option value="Key" className="bg-black">Key Icon</option>
                <option value="Terminal" className="bg-black">Terminal Icon</option>
                <option value="Download" className="bg-black">Download Icon</option>
                <option value="ShieldAlert" className="bg-black">Shield Icon</option>
             </select>
             <button type="submit" className="w-full bg-white text-black p-2 rounded font-bold">Add Tutorial</button>
           </form>
        </div>

        <div className="lg:col-span-2 space-y-4">
           {tutorials.map((t) => (
             <div key={t.id} className="bg-white/5 border border-white/10 rounded-xl p-6 flex justify-between items-center">
                <div>
                   <h3 className="font-bold">{t.title}</h3>
                   <p className="text-sm text-gray-400">{t.description}</p>
                   <a href={t.url} target="_blank" className="text-xs text-blue-400">{t.url}</a>
                </div>
                <button onClick={() => handleDelete(t.id)} className="bg-red-500/20 text-red-400 px-3 py-1 rounded hover:bg-red-500/30">
                  Delete
                </button>
             </div>
           ))}
        </div>
      </div>
    </div>
  );
}
