'use client';

import { useState, useEffect } from 'react';
import { useToast } from '@/app/context/ToastContext';
import { Plus, Trash2, Sparkles } from 'lucide-react';

export default function AdminServicesPage() {
  const [services, setServices] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [formData, setFormData] = useState({
    title: '',
    slug: '',
    image: '',
    description: '',
    priceRange: '',
    features: ''
  });
  const { addToast } = useToast();

  useEffect(() => {
    fetchServices();
  }, []);

  const fetchServices = async () => {
    try {
      const res = await fetch('/api/services');
      const data = await res.json();
      setServices(data.services || []);
      setLoading(false);
    } catch (error) {
      console.error(error);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
        const res = await fetch('/api/admin/services', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                ...formData,
                features: formData.features.split('\n').filter(f => f.trim() !== '')
            })
        });

        if (res.ok) {
            addToast("Service created", 'success');
            setFormData({ title: '', slug: '', image: '', description: '', priceRange: '', features: '' });
            fetchServices();
        } else {
            addToast("Failed to create service", 'error');
        }
    } catch (error) {
        addToast("Error creating service", 'error');
    }
  };

  const handleDelete = async (id: string) => {
      if(!confirm("Delete this service?")) return;
      try {
        const res = await fetch('/api/admin/services', {
            method: 'DELETE',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ id })
        });
        if(res.ok) {
            addToast("Service deleted", 'success');
            fetchServices();
        }
      } catch (e) {
          addToast("Error deleting", 'error');
      }
  }

  return (
    <div className="p-6">
      <h1 className="text-3xl font-bold mb-8 flex items-center gap-2">
        <Sparkles className="text-purple-500" /> Manage Services
      </h1>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* Create Form */}
        <div className="bg-white/5 border border-white/10 rounded-xl p-6 h-fit">
            <h2 className="text-xl font-bold mb-6">Add New Service</h2>
            <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                    <label className="block text-sm font-medium text-gray-400 mb-1">Title</label>
                    <input 
                        type="text" 
                        value={formData.title}
                        onChange={e => setFormData({...formData, title: e.target.value})}
                        className="w-full bg-black/50 border border-white/10 rounded-lg px-3 py-2 text-white"
                        placeholder="e.g. Leveling Service"
                        required
                    />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-400 mb-1">Slug (URL)</label>
                    <input 
                        type="text" 
                        value={formData.slug}
                        onChange={e => setFormData({...formData, slug: e.target.value})}
                        className="w-full bg-black/50 border border-white/10 rounded-lg px-3 py-2 text-white"
                        placeholder="e.g. leveling-service"
                        required
                    />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-400 mb-1">Image URL</label>
                    <input 
                        type="text" 
                        value={formData.image}
                        onChange={e => setFormData({...formData, image: e.target.value})}
                        className="w-full bg-black/50 border border-white/10 rounded-lg px-3 py-2 text-white"
                        placeholder="https://example.com/image.png"
                    />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-400 mb-1">Price Range</label>
                    <input 
                        type="text" 
                        value={formData.priceRange}
                        onChange={e => setFormData({...formData, priceRange: e.target.value})}
                        className="w-full bg-black/50 border border-white/10 rounded-lg px-3 py-2 text-white"
                        placeholder="e.g. $5 - $100"
                        required
                    />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-400 mb-1">Description</label>
                    <textarea 
                        value={formData.description}
                        onChange={e => setFormData({...formData, description: e.target.value})}
                        className="w-full bg-black/50 border border-white/10 rounded-lg px-3 py-2 text-white"
                        rows={3}
                        placeholder="Service description..."
                        required
                    />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-400 mb-1">Features (One per line)</label>
                    <textarea 
                        value={formData.features}
                        onChange={e => setFormData({...formData, features: e.target.value})}
                        className="w-full bg-black/50 border border-white/10 rounded-lg px-3 py-2 text-white"
                        rows={4}
                        placeholder="Fast Delivery&#10;Safe Method&#10;24/7 Support"
                    />
                </div>
                <button type="submit" className="w-full bg-purple-600 hover:bg-purple-500 text-white font-bold py-2 rounded-lg transition-colors">
                    Create Service
                </button>
            </form>
        </div>

        {/* List */}
        <div className="lg:col-span-2 space-y-4">
            <h2 className="text-xl font-bold mb-4">Active Services</h2>
            {services.length === 0 ? (
                <div className="text-gray-500">No services found.</div>
            ) : (
                services.map(service => (
                    <div key={service.id} className="bg-white/5 border border-white/10 rounded-xl p-6 flex justify-between items-start group">
                        <div>
                            <h3 className="text-lg font-bold text-white">{service.title}</h3>
                            <span className="text-xs bg-purple-500/20 text-purple-400 px-2 py-0.5 rounded border border-purple-500/20">
                                {service.priceRange}
                            </span>
                            <p className="text-gray-400 text-sm mt-2 max-w-md">{service.description}</p>
                            <div className="flex gap-2 mt-3 flex-wrap">
                                {service.features.map((f: string, i: number) => (
                                    <span key={i} className="text-xs text-gray-500 bg-white/5 px-2 py-1 rounded">
                                        {f}
                                    </span>
                                ))}
                            </div>
                        </div>
                        <button 
                            onClick={() => handleDelete(service.id)}
                            className="p-2 text-gray-500 hover:text-red-500 hover:bg-red-500/10 rounded-lg transition-colors"
                        >
                            <Trash2 size={18} />
                        </button>
                    </div>
                ))
            )}
        </div>
      </div>
    </div>
  );
}
