'use client';

import { useState, useEffect, useRef } from 'react';
import { useToast } from '@/app/context/ToastContext';
import { Send, Inbox, DollarSign, CheckCircle, XCircle, Clock } from 'lucide-react';

export default function AdminRequestsPage() {
  const [requests, setRequests] = useState<any[]>([]);
  const [selectedRequest, setSelectedRequest] = useState<any>(null);
  const [messages, setMessages] = useState<any[]>([]);
  const [input, setInput] = useState('');
  const [priceInput, setPriceInput] = useState('');
  const [loading, setLoading] = useState(true);
  const chatRef = useRef<HTMLDivElement>(null);
  const { addToast } = useToast();

  // Fetch ALL requests
  const fetchRequests = async () => {
    const res = await fetch('/api/inbox'); // Re-using inbox API which returns all for admin
    const data = await res.json();
    setRequests(data.requests || []);
    setLoading(false);
  };

  useEffect(() => {
    fetchRequests();
  }, []);

  // Fetch chat
  useEffect(() => {
    if (!selectedRequest) return;
    
    const fetchChat = async () => {
        const res = await fetch(`/api/services/chat?requestId=${selectedRequest.id}`);
        const data = await res.json();
        setMessages(data.messages || []);
    };

    fetchChat();
    const interval = setInterval(fetchChat, 3000);
    return () => clearInterval(interval);
  }, [selectedRequest]);

  useEffect(() => {
    if (chatRef.current) {
        chatRef.current.scrollTop = chatRef.current.scrollHeight;
    }
  }, [messages]);

  const handleSend = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!input.trim() || !selectedRequest) return;

    try {
        await fetch('/api/services/chat', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                requestId: selectedRequest.id,
                message: input
            })
        });
        setInput('');
    } catch (error) {
        addToast("Failed to send", 'error');
    }
  };

  const handleSetPrice = async () => {
      if (!priceInput || isNaN(Number(priceInput))) {
          addToast("Invalid price", 'error');
          return;
      }
      if (!confirm(`Request payment of $${priceInput}?`)) return;

      try {
          const res = await fetch(`/api/admin/requests/${selectedRequest.id}/offer`, {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ price: Number(priceInput) })
          });
          if (res.ok) {
              addToast("Payment request sent", 'success');
              fetchRequests(); // Refresh list status
              // Update local selected request status to reflect change immediately
              setSelectedRequest((prev: any) => ({...prev, status: 'awaiting_payment', agreedPrice: Number(priceInput)}));
          } else {
              addToast("Failed to send request", 'error');
          }
      } catch (e) {
          addToast("Error", 'error');
      }
  };

  const handleStatusUpdate = async (status: string) => {
      if (!confirm(`Mark request as ${status.replace('_', ' ')}?`)) return;
      
      try {
        const res = await fetch(`/api/admin/requests/${selectedRequest.id}/status`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ status })
        });
        if (res.ok) {
            addToast("Status updated", 'success');
            fetchRequests();
            setSelectedRequest((prev: any) => ({...prev, status }));
        }
      } catch (e) {
          addToast("Error", 'error');
      }
  };

  if (loading) return <div className="p-8 text-center text-gray-500">Loading...</div>;

  return (
    <div className="max-w-7xl mx-auto px-4 py-8 h-[calc(100vh-5rem)] flex gap-6">
      {/* Requests List */}
      <div className="w-1/3 bg-white/5 border border-white/10 rounded-2xl overflow-hidden flex flex-col">
        <div className="p-4 bg-black/20 border-b border-white/10">
            <h2 className="font-bold text-white">Active Requests</h2>
        </div>
        <div className="flex-1 overflow-y-auto">
            {requests.map(req => (
                <button
                    key={req.id}
                    onClick={() => setSelectedRequest(req)}
                    className={`w-full text-left p-4 border-b border-white/5 hover:bg-white/5 ${selectedRequest?.id === req.id ? 'bg-white/10 border-l-4 border-l-pink-500' : ''}`}
                >
                    <div className="flex justify-between items-center mb-1">
                        <span className="font-bold text-white">{req.userEmail.split('@')[0]}</span>
                        <span className={`text-[10px] px-2 py-0.5 rounded uppercase ${
                            req.status === 'pending' ? 'bg-yellow-500/20 text-yellow-400' :
                            req.status === 'awaiting_payment' ? 'bg-blue-500/20 text-blue-400' :
                            req.status === 'in_progress' ? 'bg-purple-500/20 text-purple-400' :
                            'bg-gray-500/20 text-gray-400'
                        }`}>{req.status.replace('_', ' ')}</span>
                    </div>
                    <p className="text-xs text-gray-400">{req.serviceTitle}</p>
                </button>
            ))}
        </div>
      </div>

      {/* Chat & Controls */}
      <div className="w-2/3 bg-white/5 border border-white/10 rounded-2xl overflow-hidden flex flex-col">
        {selectedRequest ? (
            <>
                {/* Header with Controls */}
                <div className="p-4 border-b border-white/10 bg-black/20 flex justify-between items-center">
                    <div>
                        <h2 className="font-bold text-white">{selectedRequest.serviceTitle}</h2>
                        <p className="text-xs text-gray-400">{selectedRequest.userEmail}</p>
                    </div>
                    
                    {/* Admin Actions */}
                    <div className="flex items-center gap-2">
                        {selectedRequest.status !== 'completed' && selectedRequest.status !== 'cancelled' && (
                            <>
                                <div className="flex items-center bg-white/5 rounded-lg p-1">
                                    <span className="px-2 text-xs text-green-400">$</span>
                                    <input 
                                        type="number" 
                                        placeholder="Price"
                                        value={priceInput}
                                        onChange={e => setPriceInput(e.target.value)}
                                        className="w-16 bg-transparent text-sm text-white focus:outline-none"
                                    />
                                    <button 
                                        onClick={handleSetPrice}
                                        className="bg-green-600 hover:bg-green-500 text-white p-1.5 rounded ml-1"
                                        title="Send Payment Request"
                                    >
                                        <DollarSign size={14} />
                                    </button>
                                </div>

                                <div className="h-6 w-[1px] bg-white/10 mx-2"></div>

                                <button 
                                    onClick={() => {
                                        const txId = prompt("Enter Transaction ID (from PayPal):");
                                        if(txId) {
                                            fetch('/api/admin/requests/verify-payment', {
                                                method: 'POST',
                                                headers: { 'Content-Type': 'application/json' },
                                                body: JSON.stringify({ requestId: selectedRequest.id, transactionId: txId })
                                            }).then(res => {
                                                if(res.ok) {
                                                    addToast("Payment verified manually", 'success');
                                                    fetchRequests();
                                                    setSelectedRequest((prev: any) => ({...prev, status: 'in_progress'}));
                                                } else {
                                                    addToast("Failed to verify", 'error');
                                                }
                                            });
                                        }
                                    }}
                                    className="bg-yellow-600 hover:bg-yellow-500 text-white p-2 rounded-lg text-xs font-bold"
                                    title="Manually Verify Payment"
                                >
                                    Force Pay
                                </button>
                                <button 
                                    onClick={() => handleStatusUpdate('in_progress')}
                                    className="bg-blue-600 hover:bg-blue-500 text-white p-2 rounded-lg text-xs font-bold"
                                    title="Mark In Progress (Use if paid manually)"
                                >
                                    Start Work
                                </button>
                                <button 
                                    onClick={() => handleStatusUpdate('completed')}
                                    className="bg-purple-600 hover:bg-purple-500 text-white p-2 rounded-lg text-xs font-bold"
                                >
                                    Complete
                                </button>
                            </>
                        )}
                    </div>
                </div>

                {/* Chat */}
                <div ref={chatRef} className="flex-1 overflow-y-auto p-6 space-y-4">
                    <div className="bg-white/5 border border-white/10 rounded-xl p-4 mb-6">
                        <h3 className="text-xs font-bold text-gray-500 uppercase mb-2">Request Details</h3>
                        <p className="text-gray-300 text-sm">{selectedRequest.initialDetails}</p>
                    </div>
                    {messages.map((msg) => (
                        <div key={msg.id} className={`flex ${msg.senderRole === 'system' ? 'justify-center' : msg.senderRole === 'admin' || msg.senderRole === 'staff' ? 'justify-end' : 'justify-start'}`}>
                            <div className={`max-w-[80%] p-3 rounded-2xl text-sm ${
                                msg.senderRole === 'system' ? 'bg-black/20 text-gray-500 text-xs px-4 py-1 rounded-full' :
                                msg.senderRole === 'admin' || msg.senderRole === 'staff'
                                    ? 'bg-purple-600 text-white rounded-tr-none'
                                    : 'bg-white/10 text-gray-300 rounded-tl-none'
                            }`}>
                                {msg.message}
                            </div>
                        </div>
                    ))}
                </div>

                {/* Input */}
                <form onSubmit={handleSend} className="p-4 border-t border-white/10 bg-black/20 flex gap-2">
                    <input
                        type="text"
                        value={input}
                        onChange={e => setInput(e.target.value)}
                        placeholder="Reply as Admin..."
                        className="flex-1 bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:outline-none focus:border-purple-500 transition-colors"
                    />
                    <button type="submit" className="bg-purple-600 hover:bg-purple-500 text-white p-3 rounded-xl">
                        <Send size={20} />
                    </button>
                </form>
            </>
        ) : (
            <div className="flex-1 flex items-center justify-center text-gray-500">Select a request</div>
        )}
      </div>
    </div>
  );
}
