'use client';

import { useEffect, useState } from 'react';
import { useToast } from '@/app/context/ToastContext';

export default function ManageProductsPage() {
  const [products, setProducts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [editingId, setEditingId] = useState<string | null>(null);
  
  // Form State
  const [formData, setFormData] = useState({
    slug: '',
    name: '',
    price: 0,
    description: '',
    image: '',
    status: 'available',
    paypalLink: ''
  });

  const { addToast } = useToast();

  const fetchProducts = () => {
    fetch('/api/products')
      .then((res) => res.json())
      .then((data) => {
        setProducts(data.products || []);
        setLoading(false);
      })
      .catch(() => addToast("Failed to fetch products", 'error'));
  };

  useEffect(() => {
    fetchProducts();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    const url = editingId ? `/api/products/${editingId}` : '/api/products';
    const method = editingId ? 'PUT' : 'POST';

    const res = await fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(formData)
    });

    if (res.ok) {
        addToast(editingId ? "Product updated" : "Product added", 'success');
        setFormData({ slug: '', name: '', price: 0, description: '', image: '', status: 'available', paypalLink: '' });
        setEditingId(null);
        fetchProducts();
    } else {
        const err = await res.json();
        addToast(err.error || "Action failed", 'error');
    }
  };

  const handleEdit = (p: any) => {
    setEditingId(p.id);
    setFormData({
        slug: p.slug,
        name: p.name,
        price: p.price,
        description: p.description,
        image: p.image || '',
        status: p.status,
        paypalLink: p.paypalLink || ''
    });
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const handleCancel = () => {
      setEditingId(null);
      setFormData({ slug: '', name: '', price: 0, description: '', image: '', status: 'available', paypalLink: '' });
  };

  const handleDelete = async (id: string) => {
    if(!confirm("Delete this product?")) return;
    await fetch(`/api/products/${id}`, { method: 'DELETE' });
    fetchProducts();
    addToast("Product deleted", 'info');
  };

  if (loading) return <div>Loading...</div>;

  return (
    <div className="max-w-6xl mx-auto px-4 py-12">
      <h1 className="text-3xl font-bold mb-8">Manage Products</h1>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <div className="bg-white/5 border border-white/10 rounded-xl p-6 h-fit">
           <h2 className="text-xl font-bold mb-4">{editingId ? 'Edit Product' : 'Add New Product'}</h2>
           <form onSubmit={handleSubmit} className="space-y-4">
             <div>
                <label className="text-xs text-gray-400">Slug (URL ID)</label>
                <input 
                    type="text" placeholder="e.g. final-stand-script" className="w-full bg-black/20 border border-white/10 rounded p-2"
                    value={formData.slug} onChange={e => setFormData({...formData, slug: e.target.value})} required
                    disabled={!!editingId} // Slug usually shouldn't change as it breaks links
                />
             </div>
             <div>
                <label className="text-xs text-gray-400">Name</label>
                <input 
                    type="text" placeholder="Product Name" className="w-full bg-black/20 border border-white/10 rounded p-2"
                    value={formData.name} onChange={e => setFormData({...formData, name: e.target.value})} required
                />
             </div>
             <div>
                <label className="text-xs text-gray-400">Price ($)</label>
                <input 
                    type="number" step="0.01" className="w-full bg-black/20 border border-white/10 rounded p-2"
                    value={formData.price} onChange={e => setFormData({...formData, price: parseFloat(e.target.value)})} required
                />
             </div>
             <div>
                <label className="text-xs text-gray-400">Description</label>
                <textarea 
                    placeholder="Description" className="w-full bg-black/20 border border-white/10 rounded p-2 h-24"
                    value={formData.description} onChange={e => setFormData({...formData, description: e.target.value})} required
                />
             </div>
             <div>
                <label className="text-xs text-gray-400">Image Path (e.g. /fsr.png)</label>
                <input 
                    type="text" placeholder="/image.png" className="w-full bg-black/20 border border-white/10 rounded p-2"
                    value={formData.image} onChange={e => setFormData({...formData, image: e.target.value})}
                />
             </div>
             <div>
                <label className="text-xs text-gray-400">PayPal Link (Optional)</label>
                <input 
                    type="text" placeholder="https://paypal.me/..." className="w-full bg-black/20 border border-white/10 rounded p-2"
                    value={formData.paypalLink} onChange={e => setFormData({...formData, paypalLink: e.target.value})}
                />
             </div>
             <div>
                <label className="text-xs text-gray-400">Status</label>
                <select 
                    className="w-full bg-black/20 border border-white/10 rounded p-2"
                    value={formData.status} onChange={e => setFormData({...formData, status: e.target.value})}
                >
                    <option value="available" className="bg-black">Available</option>
                    <option value="coming_soon" className="bg-black">Coming Soon</option>
                </select>
             </div>
             
             <div className="flex gap-2">
                <button type="submit" className="flex-1 bg-white text-black p-2 rounded font-bold">
                    {editingId ? 'Update Product' : 'Add Product'}
                </button>
                {editingId && (
                    <button type="button" onClick={handleCancel} className="bg-gray-600 text-white p-2 rounded font-bold">
                        Cancel
                    </button>
                )}
             </div>
           </form>
        </div>

        <div className="lg:col-span-2 space-y-4">
           {products.map((p) => (
             <div key={p.id} className="bg-white/5 border border-white/10 rounded-xl p-6 flex justify-between items-start">
                <div>
                   <h3 className="font-bold text-lg">{p.name}</h3>
                   <span className="text-xs bg-white/10 px-2 py-1 rounded text-gray-300">{p.slug}</span>
                   <p className="text-sm text-gray-400 mt-2 line-clamp-2">{p.description}</p>
                   <div className="mt-2 flex items-center gap-4 text-sm">
                       <span className="text-green-400 font-bold">${p.price}</span>
                       <span className={`uppercase text-xs font-bold ${p.status === 'available' ? 'text-blue-400' : 'text-yellow-400'}`}>
                           {p.status.replace('_', ' ')}
                       </span>
                   </div>
                </div>
                <div className="flex flex-col gap-2">
                    <button onClick={() => handleEdit(p)} className="bg-blue-500/20 text-blue-400 px-3 py-1 rounded hover:bg-blue-500/30 text-sm">
                    Edit
                    </button>
                    <button onClick={() => handleDelete(p.id)} className="bg-red-500/20 text-red-400 px-3 py-1 rounded hover:bg-red-500/30 text-sm">
                    Delete
                    </button>
                </div>
             </div>
           ))}
        </div>
      </div>
    </div>
  );
}
