'use client';

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { useToast } from '@/app/context/ToastContext';
import { Trash2, X, Check } from 'lucide-react';

export default function AdminPage() {
  const [submissions, setSubmissions] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [assignModalOpen, setAssignModalOpen] = useState(false);
  const [assignData, setAssignData] = useState({ email: '', slug: '', txId: '' });
  
  const router = useRouter();
  const { addToast } = useToast();

  useEffect(() => {
    fetch('/api/admin/submissions/list')
      .then(async (res) => {
        if (res.status === 403) {
          router.push('/');
          return;
        }
        const data = await res.json();
        setSubmissions(data.submissions || []);
        setLoading(false);
      })
      .catch((e) => {
        console.error(e);
        setLoading(false);
        addToast('Failed to load submissions', 'error');
      });
  }, [router, addToast]);

  const handleDelete = async (id: string) => {
      if(!confirm("Delete this submission? This will remove it from history.")) return;
      
      try {
          const res = await fetch(`/api/admin/submissions/${id}`, { method: 'DELETE' });
          if (res.ok) {
              setSubmissions(prev => prev.filter(s => s.id !== id));
              addToast("Submission deleted", 'success');
          } else {
              addToast("Failed to delete", 'error');
          }
      } catch (error) {
          addToast("Error deleting", 'error');
      }
  };

  const handleAssignSubmit = async (e: React.FormEvent) => {
      e.preventDefault();
      try {
          const res = await fetch('/api/admin/assign-license', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ 
                  email: assignData.email, 
                  productSlug: assignData.slug, 
                  transactionId: assignData.txId, 
                  productName: assignData.slug 
              })
          });
          const data = await res.json();
          if (res.ok) {
              addToast(`Success! Key assigned: ${data.key}`, 'success');
              setAssignModalOpen(false);
              setAssignData({ email: '', slug: '', txId: '' });
              // Refresh list to show new manual submission
              window.location.reload(); 
          } else {
              addToast(data.error || "Failed", 'error');
          }
      } catch (e) {
          addToast("Error", 'error');
      }
  };

  const handleApprove = async (id: string) => {
    if (!confirm("Approve this submission?")) return;
    
    try {
      const res = await fetch(`/api/admin/submissions/${id}/approve`, { method: 'POST', body: JSON.stringify({}) });
      
      let data;
      try {
        data = await res.json();
      } catch (e) {
        console.error("Failed to parse JSON:", e);
        addToast("Server returned an invalid response.", 'error');
        return;
      }

      if (!res.ok) {
        addToast(data.error || "Failed to approve", 'error');
        return;
      }

      // Refresh list locally
      setSubmissions((prev) => prev.map(s => s.id === id ? { ...s, status: 'approved' } : s));
      addToast("Submission approved successfully", 'success');
      
      if (data.warning) {
        addToast("Warning: " + data.warning, 'warning');
      }
    } catch (error) {
      console.error("Approval request failed:", error);
      addToast("Network error occurred while approving.", 'error');
    }
  };

  const handleReject = async (id: string) => {
    const reason = prompt("Reason for rejection:");
    if (!reason) return;
    await fetch(`/api/admin/submissions/${id}/reject`, {
      method: 'POST',
      body: JSON.stringify({ reason })
    });
    setSubmissions((prev) => prev.map(s => s.id === id ? { ...s, status: 'rejected' } : s));
    addToast("Submission rejected", 'info');
  };

  const handleAssignLicense = async () => {
      const email = prompt("Enter User Email:");
      if (!email) return;
      
      const slug = prompt("Enter Product Slug (e.g. final-stand-remastered):");
      if (!slug) return;

      const txId = prompt("Enter Transaction ID (Optional):");

      try {
          const res = await fetch('/api/admin/assign-license', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ email, productSlug: slug, transactionId: txId, productName: slug })
          });
          const data = await res.json();
          if (res.ok) {
              addToast(`Success! Key assigned: ${data.key}`, 'success');
          } else {
              addToast(data.error || "Failed", 'error');
          }
      } catch (e) {
          addToast("Error", 'error');
      }
  };

  if (loading) return <div className="min-h-screen flex items-center justify-center text-gray-400">Loading dashboard...</div>;

  return (
    <div className="max-w-7xl mx-auto px-4 py-12 relative">
      {/* Modal */}
      {assignModalOpen && (
          <div className="fixed inset-0 z-[100] flex items-center justify-center bg-black/80 backdrop-blur-sm p-4">
              <div className="bg-[#1a1a1a] border border-white/10 rounded-2xl p-8 w-full max-w-md shadow-2xl shadow-yellow-500/10 animate-in fade-in zoom-in duration-200">
                  <div className="flex justify-between items-center mb-6">
                      <h2 className="text-2xl font-bold text-white">Manual Key Delivery</h2>
                      <button onClick={() => setAssignModalOpen(false)} className="text-gray-400 hover:text-white"><X /></button>
                  </div>
                  
                  <form onSubmit={handleAssignSubmit} className="space-y-4">
                      <div>
                          <label className="block text-sm font-medium text-gray-400 mb-1">User Email</label>
                          <input 
                              type="email" 
                              value={assignData.email}
                              onChange={e => setAssignData({...assignData, email: e.target.value})}
                              className="w-full bg-black/50 border border-white/10 rounded-xl p-3 text-white focus:border-yellow-500 outline-none"
                              placeholder="customer@example.com"
                              required
                          />
                      </div>
                      <div>
                          <label className="block text-sm font-medium text-gray-400 mb-1">Product Slug</label>
                          <input 
                              type="text" 
                              value={assignData.slug}
                              onChange={e => setAssignData({...assignData, slug: e.target.value})}
                              className="w-full bg-black/50 border border-white/10 rounded-xl p-3 text-white focus:border-yellow-500 outline-none"
                              placeholder="e.g. final-stand-remastered"
                              required
                          />
                      </div>
                      <div>
                          <label className="block text-sm font-medium text-gray-400 mb-1">Transaction ID (Optional)</label>
                          <input 
                              type="text" 
                              value={assignData.txId}
                              onChange={e => setAssignData({...assignData, txId: e.target.value})}
                              className="w-full bg-black/50 border border-white/10 rounded-xl p-3 text-white focus:border-yellow-500 outline-none"
                              placeholder="PayPal ID or Manual Note"
                          />
                      </div>
                      
                      <button type="submit" className="w-full bg-yellow-600 hover:bg-yellow-500 text-black font-bold py-3 rounded-xl mt-4 transition-colors">
                          Assign Key
                      </button>
                  </form>
              </div>
          </div>
      )}

      <div className="flex justify-between items-center mb-8">
        <h1 className="text-3xl font-bold">Admin Dashboard</h1>
        <div className="space-x-4">
          <button onClick={() => setAssignModalOpen(true)} className="bg-yellow-600 hover:bg-yellow-500 text-black font-bold px-4 py-2 rounded-lg transition-colors shadow-lg shadow-yellow-500/20">
             + Manual Give Key
          </button>
          <a href="/admin/licenses" className="bg-blue-600 hover:bg-blue-500 text-white px-4 py-2 rounded-lg transition-colors">
             Licenses
          </a>
          <a href="/admin/tutorials" className="bg-blue-600 hover:bg-blue-500 text-white px-4 py-2 rounded-lg transition-colors">
             Tutorials
          </a>
          <a href="/admin/updates" className="bg-blue-600 hover:bg-blue-500 text-white px-4 py-2 rounded-lg transition-colors">
             Updates
          </a>
          <a href="/admin/users" className="bg-purple-600 hover:bg-purple-500 text-white px-4 py-2 rounded-lg transition-colors">
             Users
          </a>
          <a href="/admin/products" className="bg-green-600 hover:bg-green-500 text-white px-4 py-2 rounded-lg transition-colors">
             Products
          </a>
          <a href="/admin/services" className="bg-pink-600 hover:bg-pink-500 text-white px-4 py-2 rounded-lg transition-colors">
             Services
          </a>
          <a href="/admin/requests" className="bg-orange-600 hover:bg-orange-500 text-white px-4 py-2 rounded-lg transition-colors">
             Requests
          </a>
        </div>
      </div>
      
      {submissions.length === 0 ? (
        <div className="text-gray-500 text-center py-12 border border-white/10 rounded-xl bg-white/5">No submissions found.</div>
      ) : (
        <div className="overflow-x-auto bg-white/5 border border-white/10 rounded-xl">
          <table className="w-full text-left text-sm">
            <thead className="bg-black/20 text-gray-400 uppercase font-medium">
              <tr>
                <th className="px-6 py-4 whitespace-nowrap">Date</th>
                <th className="px-6 py-4 whitespace-nowrap">User</th>
                <th className="px-6 py-4 whitespace-nowrap">Product</th>
                <th className="px-6 py-4 whitespace-nowrap">Amount</th>
                <th className="px-6 py-4 whitespace-nowrap">Trans ID</th>
                <th className="px-6 py-4 whitespace-nowrap">Status</th>
                <th className="px-6 py-4 whitespace-nowrap">Receipt</th>
                <th className="px-6 py-4 whitespace-nowrap">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-white/5">
              {submissions.map((sub) => (
                <tr key={sub.id} className="hover:bg-white/5 transition-colors">
                  <td className="px-6 py-4 text-gray-400 whitespace-nowrap">{new Date(sub.createdAt).toLocaleDateString()} {new Date(sub.createdAt).toLocaleTimeString()}</td>
                  <td className="px-6 py-4 whitespace-nowrap">{sub.userEmail}</td>
                  <td className="px-6 py-4 whitespace-nowrap">{sub.productName}</td>
                  <td className="px-6 py-4 whitespace-nowrap">${sub.amount}</td>
                  <td className="px-6 py-4 font-mono text-xs whitespace-nowrap">{sub.transactionId}</td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`px-2 py-1 rounded text-xs font-bold uppercase ${
                      sub.status === 'approved' ? 'bg-green-500/20 text-green-400' :
                      sub.status === 'rejected' ? 'bg-red-500/20 text-red-400' :
                      'bg-yellow-500/20 text-yellow-400'
                    }`}>
                      {sub.status}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    {sub.receiptFilename ? (
                      <a 
                        href={`/api/admin/receipts/${sub.receiptFilename}`} 
                        target="_blank" 
                        className="text-blue-400 hover:underline"
                        rel="noreferrer"
                      >
                        View Receipt
                      </a>
                    ) : <span className="text-gray-600">None</span>}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="flex gap-2">
                      {sub.status === 'pending' && (
                        <>
                          <button 
                            onClick={() => handleApprove(sub.id)}
                            className="bg-green-600 hover:bg-green-500 text-white px-3 py-1 rounded text-xs transition-colors"
                          >
                            Approve
                          </button>
                          <button 
                            onClick={() => handleReject(sub.id)}
                            className="bg-red-600 hover:bg-red-500 text-white px-3 py-1 rounded text-xs transition-colors"
                          >
                            Reject
                          </button>
                        </>
                      )}
                      <button 
                        onClick={() => handleDelete(sub.id)}
                        className="p-1 text-gray-500 hover:text-red-500 transition-colors"
                        title="Delete"
                      >
                        <Trash2 size={16} />
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}
    </div>
  );
}
