'use client';

import { useEffect, useState } from 'react';
import { useToast } from '@/app/context/ToastContext';

export default function LicensesPage() {
  const [licenses, setLicenses] = useState<any[]>([]);
  const [products, setProducts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [newKeys, setNewKeys] = useState('');
  const [selectedProduct, setSelectedProduct] = useState('');
  const { addToast } = useToast();

  const fetchData = async () => {
    try {
      const [licRes, prodRes] = await Promise.all([
        fetch('/api/admin/licenses/list'),
        fetch('/api/products')
      ]);

      const licData = await licRes.json();
      const prodData = await prodRes.json();

      if (licData.licenses) setLicenses(licData.licenses);
      if (prodData.products) {
          setProducts(prodData.products);
          if (prodData.products.length > 0 && !selectedProduct) {
              setSelectedProduct(prodData.products[0].slug);
          }
      }
      setLoading(false);
    } catch (error) {
      addToast("Failed to load data", 'error');
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  const handleAddKeys = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedProduct) {
        addToast("Please select a product", 'error');
        return;
    }

    const keys = newKeys.split('\n').filter(k => k.trim());
    
    await fetch('/api/admin/licenses/add', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        productSlug: selectedProduct,
        keys
      })
    });

    setNewKeys('');
    fetchData(); // Refresh list
    addToast(`Added ${keys.length} keys successfully`, 'success');
  };

  const availableCount = licenses.filter(l => l.status === 'available').length;
  const usedCount = licenses.filter(l => l.status === 'used').length;

  if (loading) return <div className="text-gray-400">Loading licenses...</div>;

  return (
    <div className="space-y-8">
      <div className="bg-white/5 border border-white/10 rounded-xl p-6">
        <h2 className="text-xl font-bold mb-4">Add New Licenses</h2>
        <form onSubmit={handleAddKeys} className="space-y-4">
          <div>
            <label className="block text-sm text-gray-400 mb-1">Product</label>
            <select 
              value={selectedProduct}
              onChange={(e) => setSelectedProduct(e.target.value)}
              className="w-full bg-black/20 border border-white/10 rounded-lg px-4 py-2 text-white focus:outline-none focus:border-white/30"
            >
              {products.length === 0 && <option value="">No products found</option>}
              {products.map(p => (
                <option key={p.id} value={p.slug} className="bg-black">{p.name}</option>
              ))}
            </select>
          </div>
          
          <div>
             <label className="block text-sm text-gray-400 mb-1">Keys (One per line)</label>
             <textarea
               value={newKeys}
               onChange={(e) => setNewKeys(e.target.value)}
               className="w-full bg-black/20 border border-white/10 rounded-lg px-4 py-2 text-white focus:outline-none focus:border-white/30 h-32 font-mono text-sm"
               placeholder="XXXX-XXXX-XXXX-XXXX"
             />
          </div>
          
          <button type="submit" className="bg-white text-black px-4 py-2 rounded-lg font-medium hover:bg-gray-200">
            Add Keys
          </button>
        </form>
      </div>

      <div className="bg-white/5 border border-white/10 rounded-xl overflow-hidden">
        <div className="p-4 border-b border-white/10 flex justify-between items-center">
           <h2 className="font-bold">License Inventory</h2>
           <div className="text-sm space-x-4">
             <span className="text-green-400">Available: {availableCount}</span>
             <span className="text-red-400">Used: {usedCount}</span>
           </div>
        </div>
        <div className="overflow-x-auto max-h-96">
          <table className="w-full text-left text-sm">
            <thead className="bg-black/20 text-gray-400 sticky top-0 backdrop-blur-md">
              <tr>
                <th className="px-6 py-3">Product</th>
                <th className="px-6 py-3">Key</th>
                <th className="px-6 py-3">Status</th>
                <th className="px-6 py-3">Added</th>
                <th className="px-6 py-3">Used By</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-white/5">
              {licenses.map((lic, i) => (
                <tr key={i} className="hover:bg-white/5">
                  <td className="px-6 py-3">{lic.productSlug}</td>
                  <td className="px-6 py-3 font-mono text-xs">{lic.key}</td>
                  <td className="px-6 py-3">
                    <span className={`px-2 py-1 rounded text-xs ${lic.status === 'available' ? 'bg-green-500/20 text-green-400' : 'bg-red-500/20 text-red-400'}`}>
                      {lic.status}
                    </span>
                  </td>
                  <td className="px-6 py-3 text-gray-500">{new Date(lic.createdAt).toLocaleDateString()}</td>
                  <td className="px-6 py-3 text-gray-500">{lic.usedBy || '-'}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
}
