--[[
    Roblox AI Game Scanner ("Simple AI")
    Analyzes RemoteEvents and RemoteFunctions to find interesting game data.
    
    Features:
    - Auto-hooks into network traffic (RemoteSpy)
    - "Simple AI" Heuristics to classify remotes (Economy, Admin, Combat, etc.)
    - Logs findings with AI confidence scores
    - Scans game storage for hidden values
]]

local Config = {
    AutoScroll = true,
    IgnoreChat = true,
    MinConfidence = 1 -- Minimum matches to trigger AI alert
}

-- // SERVICES //
local CoreGui = game:GetService("CoreGui")
local StarterGui = game:GetService("StarterGui")
local TweenService = game:GetService("TweenService")
local UserInputService = game:GetService("UserInputService")
local RunService = game:GetService("RunService")
local ReplicatedStorage = game:GetService("ReplicatedStorage")
local Workspace = game:GetService("Workspace")
local Players = game:GetService("Players")

-- // GUI SETUP //
local ScreenGui = Instance.new("ScreenGui")
ScreenGui.Name = "AIScanner"

-- improved parenting logic
if pcall(function() return game:GetService("CoreGui") end) then
    ScreenGui.Parent = game:GetService("CoreGui")
else
    ScreenGui.Parent = game:GetService("Players").LocalPlayer:WaitForChild("PlayerGui")
end

-- Debug print to confirm it ran
print("AI Spy Loaded! Press F9 to see console if GUI is missing.")
warn("Gui Parented to:", ScreenGui.Parent)


local MainFrame = Instance.new("Frame")
MainFrame.Name = "Main"
MainFrame.Size = UDim2.new(0, 500, 0, 350)
MainFrame.Position = UDim2.new(0.5, -250, 0.5, -175)
MainFrame.BackgroundColor3 = Color3.fromRGB(25, 25, 25)
MainFrame.BorderSizePixel = 0
MainFrame.Parent = ScreenGui

local TitleBar = Instance.new("Frame")
TitleBar.Name = "TitleBar"
TitleBar.Size = UDim2.new(1, 0, 0, 30)
TitleBar.BackgroundColor3 = Color3.fromRGB(35, 35, 35)
TitleBar.Parent = MainFrame

local Title = Instance.new("TextLabel")
Title.Text = "AI Data Scanner v1.0"
Title.Size = UDim2.new(1, -10, 1, 0)
Title.Position = UDim2.new(0, 10, 0, 0)
Title.BackgroundTransparency = 1
Title.TextColor3 = Color3.fromRGB(255, 255, 255)
Title.TextXAlignment = Enum.TextXAlignment.Left
Title.Font = Enum.Font.GothamBold
Title.TextSize = 14
Title.Parent = TitleBar

local LogList = Instance.new("ScrollingFrame")
LogList.Name = "LogList"
LogList.Size = UDim2.new(1, -20, 1, -80)
LogList.Position = UDim2.new(0, 10, 0, 40)
LogList.BackgroundColor3 = Color3.fromRGB(30, 30, 30)
LogList.BorderSizePixel = 0
LogList.Parent = MainFrame
LogList.CanvasSize = UDim2.new(0, 0, 0, 0)
LogList.ScrollBarThickness = 4

local UIListLayout = Instance.new("UIListLayout")
UIListLayout.Parent = LogList
UIListLayout.SortOrder = Enum.SortOrder.LayoutOrder
UIListLayout.Padding = UDim.new(0, 4)

-- // AI BRAIN //
local AI = {}
AI.Frequency = {} -- Track calls per second for spam detection
AI.Categories = {
    ["ECONOMY"] = {"money", "cash", "gold", "coin", "currency", "buy", "sell", "price", "cost", "wallet", "bank", "credit", "gem", "diamond"},
    ["ADMIN"] = {"admin", "ban", "kick", "kill", "system", "cmd", "command", "mod", "rank", "permission"},
    ["COMBAT"] = {"damage", "hit", "fire", "shoot", "attack", "kill", "gun", "weapon", "sword", "hp", "health"},
    ["MOVEMENT"] = {"teleport", "cframe", "pos", "speed", "jump", "fly", "warp"},
    ["DATA"] = {"save", "load", "data", "stat", "profile", "inventory", "item"}
}

function AI:Analyze(remote, args)
    local findings = {}
    local remoteName = remote.Name:lower()
    local now = tick()
    local argTypesList = {}
    
    -- 0. Spam Detection (Skip if args are nil/empty, means it's a passive scan)
    if #args > 0 then
        if not self.Frequency[remote] then self.Frequency[remote] = {count = 0, lastCheck = now} end
        local freq = self.Frequency[remote]
        if now - freq.lastCheck > 1 then
            freq.count = 0
            freq.lastCheck = now
        end
        freq.count = freq.count + 1
        if freq.count > 10 then return {}, {} end -- Ignore SPAM
    end
    
    -- 1. Check Name
    for category, keywords in pairs(self.Categories) do
        for _, keyword in ipairs(keywords) do
            if string.find(remoteName, keyword) then
                findings[category] = true
            end
        end
    end
    
    -- 2. Check Arguments & Signature Analysis
    local argTypes = ""
    for _, arg in ipairs(args) do
        local t = type(arg)
        table.insert(argTypesList, t)
        argTypes = argTypes .. t .. ","
        
        if t == "string" then
            local s = arg:lower()
            for category, keywords in pairs(self.Categories) do
                for _, keyword in ipairs(keywords) do
                    if string.find(s, keyword) then findings[category] = true end
                end
            end
        elseif t == "number" then
            if arg > 1000 and not string.find(remoteName, "pos") then findings["HIGH_VALUE"] = true end
        elseif t == "boolean" then
             -- Booleans often indicate toggles or success flags
        end
    end
    
    -- 3. Heuristics for Obfuscated Remotes
    if not next(findings) then
        -- If name is weird but args look like economy (String, Number)
        if (string.match(argTypes, "^string,number") or string.match(argTypes, "^number,string")) then
            findings["POTENTIAL_TRANSACTION"] = true
        end
        -- If args look like combat (Instance, Number) -> (Target, Damage)
        if string.match(argTypes, "^Instance,number") or string.match(argTypes, "^userdata,number") then
            findings["POTENTIAL_DAMAGE"] = true
        end
    end
    
    return findings, argTypesList
end

function AI:ScanGame()
    local results = 0
    SystemLog("Scanning game for hidden remotes...", Color3.fromRGB(255, 255, 0))
    
    local function Scan(parent)
        for _, v in ipairs(parent:GetDescendants()) do
            if v:IsA("RemoteEvent") or v:IsA("RemoteFunction") then
                local findings, _ = AI:Analyze(v, {}) -- Analyze name only
                if next(findings) then
                    Log(v, {"[SCANNED]"}, findings)
                    results = results + 1
                end
            end
        end
    end
    
    Scan(ReplicatedStorage)
    Scan(Workspace)
    Scan(Players.LocalPlayer)
    
    SystemLog("Scan complete. Found " .. results .. " interesting remotes.", Color3.fromRGB(100, 255, 100))
end

function AI:GetTags(findings)
    local tags = {}
    for cat, _ in pairs(findings) do
        table.insert(tags, cat)
    end
    return tags
end

-- // LOGGER //
local function GenerateScript(remote, args, mode)
    mode = mode or "once"
    local script = "-- AI Generated " .. (mode == "loop" and "Auto Farm" or "Script") .. " for: " .. remote.Name .. "\n"
    
    if mode == "loop" then
        script = script .. "getgenv().AutoFarm = true -- Execute 'getgenv().AutoFarm = false' to stop\n"
        script = script .. "while getgenv().AutoFarm do\n    "
    end

    script = script .. "local args = {\n"
    for i, v in ipairs(args) do
        local valStr = tostring(v)
        if type(v) == "string" then valStr = '"' .. v .. '"' end
        if type(v) == "Instance" then valStr = v:GetFullName() end
        script = script .. (mode == "loop" and "    " or "") .. "    [" .. i .. "] = " .. valStr .. ",\n"
    end
    script = script .. (mode == "loop" and "    " or "") .. "}\n\n"
    
    local callLine = ""
    if remote:IsA("RemoteEvent") then
        callLine = "game:GetService('ReplicatedStorage'):WaitForChild('" .. remote.Name .. "'):FireServer(unpack(args))"
    else
        callLine = "game:GetService('ReplicatedStorage'):WaitForChild('" .. remote.Name .. "'):InvokeServer(unpack(args))"
    end
    
    script = script .. (mode == "loop" and "    " or "") .. callLine
    
    if mode == "loop" then
        script = script .. "\n    task.wait(0.5) -- AI: Speed limit to prevent kick\nend"
    end
    
    -- Copy to clipboard if supported
    if setclipboard then 
        setclipboard(script)
        return true
    end
    return false
end

local function Log(remote, args, findings)
    local tags = AI:GetTags(findings)
    if #tags == 0 then return end -- Only log interesting things
    
    local Container = Instance.new("TextButton") -- Make it clickable
    Container.Size = UDim2.new(1, 0, 0, 25)
    Container.BackgroundTransparency = 1
    Container.Text = ""
    Container.Parent = LogList
    
    local Label = Instance.new("TextLabel")
    Label.Size = UDim2.new(1, -60, 1, 0)
    Label.BackgroundTransparency = 1
    Label.TextColor3 = Color3.fromRGB(200, 200, 200)
    Label.Font = Enum.Font.Code
    Label.TextSize = 12
    Label.TextXAlignment = Enum.TextXAlignment.Left
    Label.Parent = Container
    
    local tagStr = table.concat(tags, ", ")
    Label.Text = string.format("[%s] %s", tagStr, remote.Name)
    
    -- Color coding
    if findings["ADMIN"] then Label.TextColor3 = Color3.fromRGB(255, 100, 100)
    elseif findings["ECONOMY"] then Label.TextColor3 = Color3.fromRGB(100, 255, 100)
    elseif findings["COMBAT"] then Label.TextColor3 = Color3.fromRGB(255, 100, 255)
    elseif findings["HIGH_VALUE"] then Label.TextColor3 = Color3.fromRGB(255, 215, 0) -- Gold
    elseif findings["POTENTIAL_TRANSACTION"] then Label.TextColor3 = Color3.fromRGB(0, 255, 255) -- Cyan
    end
    
    -- Action Button
    local ActionBtn = Instance.new("TextButton")
    ActionBtn.Size = UDim2.new(0, 50, 0, 20)
    ActionBtn.Position = UDim2.new(1, -55, 0, 2)
    ActionBtn.BackgroundColor3 = Color3.fromRGB(60, 60, 60)
    ActionBtn.TextColor3 = Color3.fromRGB(255, 255, 255)
    ActionBtn.Text = "COPY"
    ActionBtn.TextSize = 10
    ActionBtn.Parent = Container
    
    ActionBtn.MouseButton1Click:Connect(function()
        if GenerateScript(remote, args) then
            ActionBtn.Text = "COPIED"
            wait(1)
            ActionBtn.Text = "COPY"
        else
            ActionBtn.Text = "FAIL"
        end
    end)
    
    LogList.CanvasSize = UDim2.new(0, 0, 0, UIListLayout.AbsoluteContentSize.Y)
    if Config.AutoScroll then
        LogList.CanvasPosition = Vector2.new(0, LogList.CanvasSize.Y.Offset)
    end
end

-- // HOOKING //
-- Note: This requires an executor environment (Synapse, ScriptWare, etc.)
local OriginalNameCall
local function Hook()
    if not getnamecallmethod or not hookmetamethod then
        -- Fallback for Studio/Testing: Just monitor events we can find
        local function Monitor(inst)
            if inst:IsA("RemoteEvent") or inst:IsA("RemoteFunction") then
                 -- Cannot hook natively in standard Lua without exploit API
                 -- We can only 'find' them here.
                 local findings = AI:Analyze(inst, {})
                 if next(findings) then
                     Log(inst, {}, findings)
                 end
            end
        end
        
        game.DescendantAdded:Connect(Monitor)
        for _, v in ipairs(game:GetDescendants()) do Monitor(v) end
        
        -- Add a visual note
        local Warning = Instance.new("TextLabel")
        Warning.Text = "WARNING: Full hook requires Executor (hookmetamethod)"
        Warning.Size = UDim2.new(1, 0, 0, 20)
        Warning.TextColor3 = Color3.fromRGB(255, 255, 0)
        Warning.BackgroundTransparency = 1
        Warning.Parent = LogList
        return
    end

    -- Real Hook Logic
    OriginalNameCall = hookmetamethod(game, "__namecall", function(self, ...)
        local args = {...}
        local method = getnamecallmethod()
        
        if (method == "FireServer" or method == "InvokeServer") and (self.ClassName == "RemoteEvent" or self.ClassName == "RemoteFunction") then
            task.spawn(function()
                local findings, argTypes = AI:Analyze(self, args)
                if next(findings) then
                    Log(self, args, findings)
                    -- Store for AI Context
                    LastLog = {
                        Remote = self,
                        Args = args,
                        ArgTypes = argTypes,
                        Findings = findings
                    }
                end
            end)
        end
        
        return OriginalNameCall(self, ...)
    end)
end

-- // SYSTEM LOGGING //
local function SystemLog(text, color)
    local Label = Instance.new("TextLabel")
    Label.Size = UDim2.new(1, 0, 0, 25)
    Label.BackgroundTransparency = 1
    Label.TextColor3 = color or Color3.fromRGB(100, 200, 255) -- AI Blue
    Label.Font = Enum.Font.GothamBold
    Label.TextSize = 12
    Label.TextXAlignment = Enum.TextXAlignment.Left
    Label.Text = "[AI] " .. text
    Label.Parent = LogList
    
    LogList.CanvasSize = UDim2.new(0, 0, 0, UIListLayout.AbsoluteContentSize.Y)
    if Config.AutoScroll then
        LogList.CanvasPosition = Vector2.new(0, LogList.CanvasSize.Y.Offset)
    end
end

-- // SEARCH BAR & AI COMMANDS //
local SearchBar = Instance.new("TextBox")
SearchBar.Name = "AIQuery"
SearchBar.Size = UDim2.new(1, -20, 0, 30)
SearchBar.Position = UDim2.new(0, 10, 1, -35)
SearchBar.BackgroundColor3 = Color3.fromRGB(40, 40, 40)
SearchBar.TextColor3 = Color3.fromRGB(255, 255, 255)
SearchBar.PlaceholderText = "Ask AI: 'find money', 'copy last', 'explain Remote1'..."
SearchBar.Parent = MainFrame

local LastLog = nil -- Track last captured remote for context commands

SearchBar.FocusLost:Connect(function(enterPressed)
    if not enterPressed then return end
    local query = SearchBar.Text:lower()
    SearchBar.Text = "" -- Clear input
    
    -- 1. COMMANDS (Actions)
    if string.find(query, "clear") or string.find(query, "reset") then
        for _, child in ipairs(LogList:GetChildren()) do
            if child:IsA("Frame") or child:IsA("TextButton") or child:IsA("TextLabel") then
                child:Destroy() 
            end
        end
        SystemLog("Logs cleared.", Color3.fromRGB(255, 255, 255))
        return
    end

    if string.find(query, "copy") and (string.find(query, "last") or string.find(query, "it")) then
        if LastLog then
            if GenerateScript(LastLog.Remote, LastLog.Args) then
                SystemLog("Copied script for " .. LastLog.Remote.Name .. " to clipboard.", Color3.fromRGB(100, 255, 100))
            else
                SystemLog("Failed to copy script (Executor not supported?)", Color3.fromRGB(255, 100, 100))
            end
        else
            SystemLog("No recent logs to copy.", Color3.fromRGB(255, 100, 100))
        end
        return
    end

    -- 3. GAME SCANNER
    if string.find(query, "scan") or string.find(query, "load") or string.find(query, "read") then
        AI:ScanGame()
        return
    end

    -- 2. AUTO FARM GENERATOR
    if string.find(query, "farm") or string.find(query, "loop") or string.find(query, "auto") then
         if LastLog then
             if GenerateScript(LastLog.Remote, LastLog.Args, "loop") then
                 SystemLog("Generated Auto Farm script for " .. LastLog.Remote.Name, Color3.fromRGB(100, 255, 100))
                 SystemLog("Paste it into your executor to start farming.", Color3.fromRGB(200, 255, 200))
             else
                 SystemLog("Failed to copy script.", Color3.fromRGB(255, 100, 100))
             end
         else
             SystemLog("No event to farm. Wait for one first.", Color3.fromRGB(255, 100, 100))
         end
         return
    end

    if string.find(query, "explain") then
        if LastLog then
            local info = "Analyzing " .. LastLog.Remote.Name .. "..."
            info = info .. " Args: (" .. table.concat(LastLog.ArgTypes, ", ") .. ")."
            if LastLog.Findings["POTENTIAL_TRANSACTION"] then
                 info = info .. " Looks like a Purchase or Trade."
            elseif LastLog.Findings["COMBAT"] then
                 info = info .. " Looks like Combat/Damage."
            else
                 info = info .. " Purpose unclear, but standard traffic."
            end
            SystemLog(info, Color3.fromRGB(255, 255, 0))
        else
             SystemLog("Wait for a remote to fire first.", Color3.fromRGB(255, 100, 100))
        end
        return
    end

    -- 2. SEARCH / FILTER
    local foundCount = 0
    SystemLog("Searching for '" .. query .. "'...", Color3.fromRGB(100, 200, 255))
    
    for _, child in ipairs(LogList:GetChildren()) do
        if child:IsA("TextButton") then -- Log entries are buttons now
            local text = child:FindFirstChild("TextLabel") and child.TextLabel.Text:lower() or ""
            local visible = false
            
            -- Simple Keyword Match
            if string.find(text, query) then visible = true end
            
            -- Smart Tag Match & Aliases
            if string.find(query, "money") or string.find(query, "cash") then
                if string.find(text, "economy") then visible = true end
            end
            if string.find(query, "fight") or string.find(query, "gun") or string.find(query, "kill") or string.find(query, "m1") or string.find(query, "dmg") then
                if string.find(text, "combat") or string.find(text, "damage") or string.find(text, "attack") then visible = true end
            end
            
            -- Broaden search: If query is "m1", also look for "click" or "activate"
            if query == "m1" then
                if string.find(text, "click") or string.find(text, "activate") or string.find(text, "tool") then visible = true end
            end

            child.Visible = visible
            if visible then foundCount = foundCount + 1 end
        end
    end
    
    if foundCount == 0 then
        SystemLog("No matches found. Try playing the game to generate logs.", Color3.fromRGB(255, 100, 100))
    else
        SystemLog("Found " .. foundCount .. " events.", Color3.fromRGB(100, 255, 100))
    end
end)
local dragging, dragInput, dragStart, startPos
TitleBar.InputBegan:Connect(function(input)
    if input.UserInputType == Enum.UserInputType.MouseButton1 then
        dragging = true
        dragStart = input.Position
        startPos = MainFrame.Position
    end
end)
TitleBar.InputEnded:Connect(function(input)
    if input.UserInputType == Enum.UserInputType.MouseButton1 then dragging = false end
end)
UserInputService.InputChanged:Connect(function(input)
    if dragging and input.UserInputType == Enum.UserInputType.MouseMovement then
        local delta = input.Position - dragStart
        MainFrame.Position = UDim2.new(startPos.X.Scale, startPos.X.Offset + delta.X, startPos.Y.Scale, startPos.Y.Offset + delta.Y)
    end
end)

-- // START //
print("Starting Hook...")
Hook()
print("Hook Initialized.")

